<?php

namespace App\Http\Controllers\Api\Auth;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Throwable;

class RoleController extends Controller
{
    // List all roles
    public function index()
    {
        return response()->json(Role::all());
    }

    // Create new role
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|unique:roles,name'
        ]);

        $role = Role::create([
            'name' => $request->name,
            'guard_name' => 'web',
        ]);

        return response()->json(['message' => 'Role created', 'role' => $role]);
    }

    // Show single role
    public function show($id)
    {
        $role = Role::findOrFail($id);
        return response()->json($role);
    }

    // Update role
    public function update(Request $request, $id)
    {
        $role = Role::findOrFail($id);

        // Prevent updating super-admin and admin role names
        $protectedRoles = ['super-admin', 'admin'];
        if (in_array($role->name, $protectedRoles)) {
            return response()->json([
                'status' => false,
                'message' => 'Cannot modify system roles (super-admin, admin).'
            ], 403);
        }

        $request->validate([
            'name' => 'required|string|unique:roles,name,' . $id,
            'permissions' => 'array'
        ]);

        $role->name = $request->name;
        $role->save();

        $loggedOutCount = 0;
        if ($request->has('permissions')) {
            $role->syncPermissions($request->permissions);

            // Logout all users with this role when permissions change
            $users = \App\Models\User::role($role->name)->get();
            foreach ($users as $user) {
                $loggedOutCount += $user->tokens()->count();
                $user->tokens()->delete();
            }
        }

        return response()->json([
            'status' => true,
            'message' => 'Role updated',
            'users_logged_out' => $loggedOutCount,
            'role' => $role
        ]);
    }

    // Delete role
    public function destroy($id)
    {
        $role = Role::findOrFail($id);

        // Prevent deleting super-admin and admin roles
        $protectedRoles = ['super-admin', 'admin'];
        if (in_array($role->name, $protectedRoles)) {
            return response()->json([
                'status' => false,
                'message' => 'Cannot delete system roles (super-admin, admin).'
            ], 403);
        }

        $role->delete();

        return response()->json([
            'status' => true,
            'message' => 'Role deleted successfully.'
        ]);
    }
}
