<?php

namespace App\Http\Controllers\Api;

use App\Models\Customer;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Throwable;
use Illuminate\Support\Facades\Auth;
use App\Models\User;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $query = Customer::with('employee.user');

        // Apply hierarchical filtering for non-admin users
        /** @var User|null $user */
        if ($user && !$user->hasAnyRole(['admin', 'super-admin'])) {
            $employeeDetail = $user->employeeDetail;
            if ($employeeDetail) {
                $subordinateIds = $employeeDetail->getAllSubordinateIds();
                $query->whereIn('employee_id', $subordinateIds);
            } else {
                // If user doesn't have employeeDetail, they shouldn't see any customers
                // unless they are admin/super-admin (handled above)
                $query->whereRaw('1 = 0');
            }
        }

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('proprietor_name', 'like', "%{$search}%");
            });
        }

        $customers = $query->get();

        return response()->json([
            'status' => true,
            'data' => $customers
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'proprietor_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address_street' => 'nullable|string',
            'address_city' => 'nullable|string',
            'address_state' => 'nullable|string',
            'address_zip_code' => 'nullable|string',
            'employee_id' => 'nullable|exists:employee_details,id',
            'status' => 'in:active,inactive',
            'image' => 'nullable|string',
        ]);

        try {
            $imagePath = null;
            if ($request->image) {
                // Handle Base64 Image
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    $imageName = 'customer_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('customers/' . $imageName, $image);
                    $imagePath = 'customers/' . $imageName;
                }
            }

            $customer = Customer::create([
                'name' => $request->name,
                'proprietor_name' => $request->proprietor_name,
                'phone' => $request->phone,
                'email' => $request->email,
                'address_street' => $request->address_street,
                'address_city' => $request->address_city,
                'address_state' => $request->address_state,
                'address_zip_code' => $request->address_zip_code,
                'employee_id' => $request->employee_id,
                'status' => $request->status ?? 'active',
                'image' => $imagePath
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Customer created successfully.',
                'data' => $customer
            ], 201);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to create customer.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $customer = Customer::with('employee')->find($id);

        if (!$customer) {
            return response()->json([
                'status' => false,
                'message' => 'Customer not found.'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $customer
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $customer = Customer::find($id);

        if (!$customer) {
            return response()->json([
                'status' => false,
                'message' => 'Customer not found.'
            ], 404);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'proprietor_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address_street' => 'nullable|string',
            'address_city' => 'nullable|string',
            'address_state' => 'nullable|string',
            'address_zip_code' => 'nullable|string',
            'employee_id' => 'nullable|exists:employee_details,id',
            'status' => 'in:active,inactive',
            'image' => 'nullable|string',
        ]);

        try {
            if ($request->filled('image')) {
                // Handle Base64 Image
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    // Delete old image
                    if ($customer->image && Storage::disk('public')->exists($customer->image)) {
                        Storage::disk('public')->delete($customer->image);
                    }

                    $imageName = 'customer_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('customers/' . $imageName, $image);
                    $customer->image = 'customers/' . $imageName;
                }
            }

            $customer->update([
                'name' => $request->name,
                'proprietor_name' => $request->proprietor_name,
                'phone' => $request->phone,
                'email' => $request->email,
                'address_street' => $request->address_street,
                'address_city' => $request->address_city,
                'address_state' => $request->address_state,
                'address_zip_code' => $request->address_zip_code,
                'employee_id' => $request->employee_id,
                'status' => $request->status ?? $customer->status
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Customer updated successfully.',
                'data' => $customer
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to update customer.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $customer = Customer::find($id);

        if (!$customer) {
            return response()->json([
                'status' => false,
                'message' => 'Customer not found.'
            ], 404);
        }

        try {
            // Delete Image
            if ($customer->image && Storage::disk('public')->exists($customer->image)) {
                Storage::disk('public')->delete($customer->image);
            }

            $customer->delete();

            return response()->json([
                'status' => true,
                'message' => 'Customer deleted successfully.'
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete customer.',
                'error' => $th->getMessage()
            ], 500);
        }
    }
}
