<?php

namespace App\Http\Controllers\Api;

use Throwable;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\EmployeeDetail;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class EmployeeDetailController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        /** @var User|null $user */
        $user = auth()->user();
        $query = EmployeeDetail::with(['user', 'role', 'designation', 'parent']);

        // Apply hierarchical filtering for non-admin users
        /** @var User|null $user */
        if ($user && !$user->hasAnyRole(['admin', 'super-admin'])) {
            $employeeDetail = $user->employeeDetail;
            if ($employeeDetail) {
                $subordinateIds = $employeeDetail->getAllSubordinateIds();
                $query->whereIn('id', $subordinateIds);
            } else {
                // If user doesn't have employeeDetail, they shouldn't see any employees
                $query->whereRaw('1 = 0');
            }
        }

        $employees = $query->get();
        return response()->json([
            'status' => true,
            'data' => $employees
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6',
            'designationId' => 'required|exists:designations,id',
            'roleId' => 'required|integer|exists:roles,id',
            'employeeId' => 'required|string|unique:employee_details,employee_id',
            'creditLimit' => 'nullable|numeric',
            'phone' => 'nullable|string',
            'nationalId' => 'nullable|string',
            'bloodGroup' => 'nullable|string',
            'territory' => 'nullable|string',
            'district' => 'nullable|string',
            'status' => 'in:active,inactive',
            'loginAllowed' => 'boolean',
            'image' => 'nullable|string', // Base64 string
            'parentId' => 'nullable|exists:employee_details,id'
        ]);

        try {
            DB::beginTransaction();

            // 1. Create User
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
            ]);

            // 2. Assign Role
            $role = Role::findById($request->roleId, 'web');
            $user->assignRole($role);

            // 3. Handle Image (Base64)
            $imagePath = null;
            if ($request->image) {
                // Expecting data:image/png;base64,.....
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    $imageName = 'employee_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('employees/' . $imageName, $image);
                    $imagePath = 'employees/' . $imageName;
                }
            }

            // 4. Create Employee Details
            $employee = EmployeeDetail::create([
                'user_id' => $user->id,
                'role_id' => $role->id,
                'designation_id' => $request->designationId,
                'parent_id' => $request->parentId,
                'employee_id' => $request->employeeId,
                'credit_limit' => $request->creditLimit ?? 0,
                'phone' => $request->phone,
                'national_id' => $request->nationalId,
                'blood_group' => $request->bloodGroup,
                'territory' => $request->territory,
                'district' => $request->district,
                'status' => $request->status ?? 'active',
                'login_allowed' => $request->loginAllowed ?? true,
                'image' => $imagePath
            ]);

            DB::commit();

            // Reload relationships
            // Reload relationships
            $employee->load(['user', 'role', 'designation', 'parent']);

            return response()->json([
                'status' => true,
                'message' => 'Employee created successfully.',
                'data' => $employee
            ], 201);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to create employee.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $employee = EmployeeDetail::with(['user', 'role', 'designation', 'parent'])->find($id);

        if (!$employee) {
            return response()->json([
                'status' => false,
                'message' => 'Employee details not found.'
            ], 404);
        }

        // Get Potential Supervisors based on Role Hierarchy
        $potentialSupervisors = [];
        if ($employee->role) {
            $roleName = strtolower($employee->role->name);
            $allowedSupervisorRoles = [];

            // Hierarchy Rules
            switch ($roleName) {
                case 'officer':
                    $allowedSupervisorRoles = ['manager'];
                    break;
                case 'manager':
                    $allowedSupervisorRoles = ['rsm'];
                    break;
                case 'rsm':
                    $allowedSupervisorRoles = ['admin', 'rsm'];
                    break;
                case 'admin':
                    $allowedSupervisorRoles = ['admin'];
                    break;
            }

            if (!empty($allowedSupervisorRoles)) {
                $potentialSupervisors = EmployeeDetail::whereHas('role', function ($q) use ($allowedSupervisorRoles) {
                    $q->whereIn('name', $allowedSupervisorRoles);
                })
                    ->with(['user', 'role', 'designation'])
                    ->where('id', '!=', $employee->id) // Exclude self
                    ->where('status', 'active')
                    ->get()
                    ->map(function ($emp) {
                        return [
                            'id' => $emp->id,
                            'name' => $emp->user ? $emp->user->name : 'Unknown',
                            'role' => $emp->role ? $emp->role->name : 'N/A',
                            'designation' => $emp->designation ? $emp->designation->name : 'N/A'
                        ];
                    });
            }
        }

        // Get Subordinates (employees who report to this employee)
        $subordinates = EmployeeDetail::with(['user', 'role', 'designation'])
            ->where('parent_id', $employee->id)
            ->get()
            ->map(function ($emp) {
                return [
                    'id' => $emp->id,
                    'name' => $emp->user ? $emp->user->name : 'Unknown',
                    'role' => $emp->role ? $emp->role->name : 'N/A',
                    'designation' => $emp->designation ? $emp->designation->name : 'N/A',
                    'phone' => $emp->phone,
                    'status' => $emp->status
                ];
            });

        return response()->json([
            'status' => true,
            'data' => $employee,
            'potential_supervisors' => $potentialSupervisors,
            'subordinates' => $subordinates
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $employee = EmployeeDetail::with(['user', 'role'])->find($id);

        if (!$employee) {
            return response()->json([
                'status' => false,
                'message' => 'Employee details not found.'
            ], 404);
        }

        $userId = $employee->user_id;

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $userId,
            'designationId' => 'exists:designations,id',
            'roleId' => 'nullable|integer|exists:roles,id',
            'employeeId' => 'nullable|string|unique:employee_details,employee_id,' . $id,
            'creditLimit' => 'nullable|numeric',
            'phone' => 'nullable|string',
            'nationalId' => 'nullable|string',
            'bloodGroup' => 'nullable|string',
            'territory' => 'nullable|string',
            'district' => 'nullable|string',
            'status' => 'in:active,inactive',
            'loginAllowed' => 'boolean',
            'image' => 'nullable|string',
            'parentId' => 'nullable|exists:employee_details,id'
        ]);

        try {
            DB::beginTransaction();

            // 1. Update User
            $user = User::findOrFail($userId);
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
            ]);

            // 2. Sync Role
            if ($request->filled('roleId')) {
                $role = Role::findById($request->roleId, 'web');
                $user->syncRoles($role);
                $employee->role_id = $role->id;
            }

            // 3. Handle Image
            if ($request->filled('image')) { // If new image provided
                // Expecting data:image/png;base64,.....
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    // Delete old image
                    if ($employee->image && Storage::disk('public')->exists($employee->image)) {
                        Storage::disk('public')->delete($employee->image);
                    }

                    $imageName = 'employee_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('employees/' . $imageName, $image);
                    $employee->image = 'employees/' . $imageName;
                }
            }

            // 4. Update Employee Details
            $employee->designation_id = $request->designationId ?? $employee->designation_id;
            $employee->parent_id = $request->parentId ?? $employee->parent_id;
            $employee->employee_id = $request->employeeId ?? $employee->employee_id;
            $employee->credit_limit = $request->creditLimit ?? $employee->credit_limit;
            $employee->phone = $request->phone ?? $employee->phone;
            $employee->national_id = $request->nationalId ?? $employee->national_id;
            $employee->blood_group = $request->bloodGroup ?? $employee->blood_group;
            $employee->territory = $request->territory ?? $employee->territory;
            $employee->district = $request->district ?? $employee->district;
            $employee->status = $request->status ?? $employee->status;
            $employee->login_allowed = $request->has('loginAllowed') ? $request->loginAllowed : $employee->login_allowed;

            $employee->save();

            DB::commit();

            // Reload relationships
            $employee->load(['user', 'role', 'designation', 'parent']);

            return response()->json([
                'status' => true,
                'message' => 'Employee details updated successfully.',
                'data' => $employee
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to update employee.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $employee = EmployeeDetail::find($id);

        if (!$employee) {
            return response()->json([
                'status' => false,
                'message' => 'Employee details not found.'
            ], 404);
        }

        try {
            DB::beginTransaction();

            // 1. Delete Image
            if ($employee->image && Storage::disk('public')->exists($employee->image)) {
                Storage::disk('public')->delete($employee->image);
            }

            // 2. Delete User (Cascade will delete EmployeeDetail)
            $user = User::findOrFail($employee->user_id);
            $user->delete();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Employee and associated user deleted successfully.'
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete employee.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Assign or update supervisor for an employee.
     */
    public function assignSupervisor(Request $request, $id)
    {
        $employee = EmployeeDetail::find($id);

        if (!$employee) {
            return response()->json([
                'status' => false,
                'message' => 'Employee not found.'
            ], 404);
        }

        $request->validate([
            'supervisorId' => 'nullable|exists:employee_details,id'
        ]);

        try {
            // Prevent self-assignment
            if ($request->supervisorId && $request->supervisorId == $employee->id) {
                return response()->json([
                    'status' => false,
                    'message' => 'An employee cannot be their own supervisor.'
                ], 422);
            }

            // Role-based supervisor validation
            if ($request->supervisorId) {
                $supervisor = EmployeeDetail::with('role')->where('id', $request->supervisorId)->first();
                $employee->load('role');

                if (!$supervisor) {
                    return response()->json([
                        'status' => false,
                        'message' => 'Supervisor not found.'
                    ], 404);
                }

                // Validate role hierarchy
                $validationResult = $this->validateRoleHierarchy($employee, $supervisor);
                if (!$validationResult['valid']) {
                    return response()->json([
                        'status' => false,
                        'message' => $validationResult['message']
                    ], 422);
                }

                // Prevent circular hierarchy
                if ($this->hasCircularHierarchy($supervisor, $employee->id)) {
                    return response()->json([
                        'status' => false,
                        'message' => 'This assignment would create a circular hierarchy.'
                    ], 422);
                }
            }

            $employee->parent_id = $request->supervisorId;
            $employee->save();

            // Reload relationships
            $employee->load(['user', 'role', 'designation', 'parent']);

            return response()->json([
                'status' => true,
                'message' => $request->supervisorId
                    ? 'Supervisor assigned successfully.'
                    : 'Supervisor removed successfully.',
                'data' => $employee
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to assign supervisor.',
                'error' => $th->getMessage()
            ], 500);
        }
    }


    /**
     * Validate role hierarchy for supervisor assignment.
     * Rules:
     * - Officer → must have Manager as supervisor
     * - Manager → must have RSM as supervisor
     * - RSM → can have any supervisor or none
     * - Admin → can have any supervisor or none
     */
    private function validateRoleHierarchy($employee, $supervisor)
    {
        $employeeRoleName = $employee->role ? strtolower($employee->role->name) : null;
        $supervisorRoleName = $supervisor->role ? strtolower($supervisor->role->name) : null;

        // If no role assigned, allow any supervisor
        if (!$employeeRoleName || !$supervisorRoleName) {
            return [
                'valid' => true,
                'message' => ''
            ];
        }

        // Role hierarchy rules
        $roleHierarchy = [
            'officer' => ['manager'],  // Officer can only have Manager as supervisor
            'manager' => ['rsm'],      // Manager can only have RSM as supervisor
            'rsm' => ['admin', 'rsm'], // RSM can have Admin or RSM as supervisor
            'admin' => ['admin']       // Admin can have Admin as supervisor
        ];

        // Check if employee role has specific supervisor requirements
        if (isset($roleHierarchy[$employeeRoleName])) {
            $allowedSupervisorRoles = $roleHierarchy[$employeeRoleName];

            if (!in_array($supervisorRoleName, $allowedSupervisorRoles)) {
                $allowedRolesStr = implode(', ', array_map('ucfirst', $allowedSupervisorRoles));
                return [
                    'valid' => false,
                    'message' => ucfirst($employeeRoleName) . ' can only have ' . $allowedRolesStr . ' as supervisor.'
                ];
            }
        }

        return [
            'valid' => true,
            'message' => ''
        ];
    }

    /**
     * Check for circular hierarchy.
     * Checks if assigning a supervisor would create a circular reference.
     * Example: A → B → C → A (circular)
     */
    private function hasCircularHierarchy($supervisor, $targetEmployeeId, $depth = 0)
    {
        // Prevent infinite loops (max depth 10 levels)
        if ($depth > 10) {
            return true;
        }

        // If supervisor doesn't exist or has no parent, no circular reference
        if (!$supervisor || !$supervisor->parent_id) {
            return false;
        }

        // If supervisor's parent is the target employee, circular reference detected
        if ($supervisor->parent_id == $targetEmployeeId) {
            return true;
        }

        // Recursively check the next level up
        $nextSupervisor = EmployeeDetail::where('id', $supervisor->parent_id)->first();
        return $this->hasCircularHierarchy($nextSupervisor, $targetEmployeeId, $depth + 1);
    }
}
