<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Order;

class ReportController extends Controller
{
    public function customerSalesReport(Request $request)
    {
        $qtySubquery = DB::table('order_items')
            ->select('order_id', DB::raw('SUM(quantity) as order_qty'))
            ->groupBy('order_id');

        $query = Order::query()
            ->join('customers', 'orders.customer_id', '=', 'customers.id')
            ->leftJoinSub($qtySubquery, 'order_quantities', function ($join) {
                $join->on('orders.id', '=', 'order_quantities.order_id');
            })
            ->select(
                'customers.id',
                'customers.name',
                'customers.phone',
                'customers.email',
                DB::raw('COUNT(orders.id) as total_orders'),
                DB::raw('SUM(orders.total_amount) as total_spent'),
                DB::raw('COALESCE(SUM(order_quantities.order_qty), 0) as total_quantity'),
                DB::raw('MAX(orders.order_date) as last_order_date')
            )
            ->groupBy('customers.id', 'customers.name', 'customers.phone', 'customers.email');

        if ($request->has('start_date')) {
            $query->whereDate('orders.order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('orders.order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }

    public function productSalesReport(Request $request)
    {
        $query = DB::table('order_items')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->select(
                'products.id',
                'products.name',
                'products.pack_size',
                DB::raw('COUNT(DISTINCT orders.id) as total_orders'),
                DB::raw('SUM(order_items.quantity) as total_quantity'),
                DB::raw('SUM(order_items.total_price) as total_spent'),
                DB::raw('AVG(order_items.unit_price) as avg_price')
            )
            ->groupBy('products.id', 'products.name', 'products.pack_size');

        if ($request->has('start_date')) {
            $query->whereDate('orders.order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('orders.order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }

    public function categorySalesReport(Request $request)
    {
        $query = DB::table('order_items')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->select(
                'categories.id',
                'categories.name',
                DB::raw('COUNT(DISTINCT orders.id) as total_orders'),
                DB::raw('SUM(order_items.quantity) as total_quantity'),
                DB::raw('SUM(order_items.total_price) as total_spent')
            )
            ->groupBy('categories.id', 'categories.name');

        if ($request->has('start_date')) {
            $query->whereDate('orders.order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('orders.order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }

    public function officerSalesReport(Request $request)
    {
        $query = DB::table('orders')
            ->join('employee_details', 'orders.employee_id', '=', 'employee_details.id')
            ->join('users', 'employee_details.user_id', '=', 'users.id')
            ->join('roles', 'employee_details.role_id', '=', 'roles.id')
            ->where('roles.name', 'officer')
            ->select(
                'employee_details.id',
                'users.name',
                'employee_details.phone',
                'employee_details.territory',
                DB::raw('COUNT(orders.id) as total_orders'),
                DB::raw('SUM(orders.total_amount) as total_spent'),
                DB::raw('SUM((SELECT SUM(quantity) FROM order_items WHERE order_id = orders.id)) as total_quantity'),
                DB::raw('MAX(orders.order_date) as last_order_date')
            )
            ->groupBy('employee_details.id', 'users.name', 'employee_details.phone', 'employee_details.territory');

        if ($request->has('start_date')) {
            $query->whereDate('orders.order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('orders.order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }

    public function managerSalesReport(Request $request)
    {
        // Managers manage Officers
        $query = DB::table('employee_details as managers')
            ->join('users as manager_users', 'managers.user_id', '=', 'manager_users.id')
            ->join('roles as manager_roles', 'managers.role_id', '=', 'manager_roles.id')
            ->join('employee_details as officers', 'officers.parent_id', '=', 'managers.id')
            ->join('orders', 'orders.employee_id', '=', 'officers.id')
            ->where('manager_roles.name', 'manager')
            ->select(
                'managers.id',
                'manager_users.name',
                'managers.phone',
                'managers.territory',
                DB::raw('COUNT(DISTINCT orders.id) as total_orders'),
                DB::raw('SUM(orders.total_amount) as total_spent'),
                DB::raw('SUM((SELECT SUM(quantity) FROM order_items WHERE order_id = orders.id)) as total_quantity'),
                DB::raw('MAX(orders.order_date) as last_order_date')
            )
            ->groupBy('managers.id', 'manager_users.name', 'managers.phone', 'managers.territory');

        if ($request->has('start_date')) {
            $query->whereDate('orders.order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('orders.order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }

    public function rsmSalesReport(Request $request)
    {
        // RSMs manage Managers who manage Officers
        // We aggregate sales of all officers under the RSM's hierarchy
        $query = DB::table('employee_details as rsms')
            ->join('users as rsm_users', 'rsms.user_id', '=', 'rsm_users.id')
            ->join('roles as rsm_roles', 'rsms.role_id', '=', 'rsm_roles.id')
            ->join('employee_details as managers', 'managers.parent_id', '=', 'rsms.id')
            ->join('employee_details as officers', 'officers.parent_id', '=', 'managers.id')
            ->join('orders', 'orders.employee_id', '=', 'officers.id')
            ->where('rsm_roles.name', 'rsm')
            ->select(
                'rsms.id',
                'rsm_users.name',
                'rsms.phone',
                'rsms.territory',
                DB::raw('COUNT(DISTINCT orders.id) as total_orders'),
                DB::raw('SUM(orders.total_amount) as total_spent'),
                DB::raw('SUM((SELECT SUM(quantity) FROM order_items WHERE order_id = orders.id)) as total_quantity'),
                DB::raw('MAX(orders.order_date) as last_order_date')
            )
            ->groupBy('rsms.id', 'rsm_users.name', 'rsms.phone', 'rsms.territory');

        if ($request->has('start_date')) {
            $query->whereDate('orders.order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('orders.order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }

    public function otherEmployeesSalesReport(Request $request)
    {
        $query = DB::table('orders')
            ->join('employee_details', 'orders.employee_id', '=', 'employee_details.id')
            ->join('users', 'employee_details.user_id', '=', 'users.id')
            ->join('roles', 'employee_details.role_id', '=', 'roles.id')
            ->whereNotIn('roles.name', ['officer', 'manager', 'rsm'])
            ->select(
                'employee_details.id',
                'users.name',
                'employee_details.phone',
                'employee_details.territory',
                DB::raw('COUNT(orders.id) as total_orders'),
                DB::raw('SUM(orders.total_amount) as total_spent'),
                DB::raw('SUM((SELECT SUM(quantity) FROM order_items WHERE order_id = orders.id)) as total_quantity'),
                DB::raw('MAX(orders.order_date) as last_order_date')
            )
            ->groupBy('employee_details.id', 'users.name', 'employee_details.phone', 'employee_details.territory');

        if ($request->has('start_date')) {
            $query->whereDate('orders.order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('orders.order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }
    public function paymentMethodReport(Request $request)
    {
        $qtySubquery = DB::table('order_items')
            ->select('order_id', DB::raw('SUM(quantity) as order_qty'))
            ->groupBy('order_id');

        $query = Order::query()
            ->leftJoinSub($qtySubquery, 'order_quantities', function ($join) {
                $join->on('orders.id', '=', 'order_quantities.order_id');
            })
            ->select(
                'payment_method as name',
                DB::raw('COUNT(orders.id) as total_orders'),
                DB::raw('SUM(orders.total_amount) as total_spent'),
                DB::raw('COALESCE(SUM(order_quantities.order_qty), 0) as total_quantity')
            )
            ->groupBy('payment_method');

        if ($request->has('start_date')) {
            $query->whereDate('order_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('order_date', '<=', $request->end_date);
        }

        $reports = $query->orderByDesc('total_spent')->get();

        return response()->json([
            'status' => true,
            'data' => $reports
        ]);
    }
}
